<?php
/**
 * Admin Order View & Management
 */
define('ECOMMERCE_ACCESS', true);
require_once '../config/db.php';
require_once '../config/auth.php';

requireAdmin();

$settings = getSettings();

// Get order ID
$orderId = intval($_GET['id'] ?? 0);
if (!$orderId) {
    header('Location: orders.php');
    exit;
}

// Get order
$order = fetchOne("
    SELECT o.*, u.name as user_name, u.email as user_email, u.phone as user_phone
    FROM orders o 
    LEFT JOIN users u ON o.user_id = u.id 
    WHERE o.id = ?
", [$orderId]);

if (!$order) {
    setFlash('error', 'Order not found');
    header('Location: orders.php');
    exit;
}

// Get order items
$orderItems = fetchAll("
    SELECT oi.*, p.image as product_image, p.slug as product_slug
    FROM order_items oi
    LEFT JOIN products p ON oi.product_id = p.id
    WHERE oi.order_id = ?
", [$orderId]);

// Get payment info
$payment = fetchOne("SELECT * FROM payments WHERE order_id = ? ORDER BY created_at DESC LIMIT 1", [$orderId]);

// Handle status update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        setFlash('error', 'Invalid request');
    } else {
        $action = $_POST['action'] ?? '';
        
        if ($action === 'update_status') {
            $newStatus = sanitize($_POST['status'] ?? '');
            $validStatuses = ['pending', 'paid', 'processing', 'shipped', 'delivered', 'cancelled', 'refunded'];
            
            if (in_array($newStatus, $validStatuses)) {
                query("UPDATE orders SET status = ? WHERE id = ?", [$newStatus, $orderId]);
                
                // If cancelled or refunded, restore stock
                if (($newStatus === 'cancelled' || $newStatus === 'refunded') && 
                    $order['status'] !== 'cancelled' && $order['status'] !== 'refunded') {
                    foreach ($orderItems as $item) {
                        if ($item['product_id']) {
                            query("UPDATE products SET stock = stock + ? WHERE id = ?", 
                                [$item['quantity'], $item['product_id']]);
                        }
                    }
                }
                
                setFlash('success', 'Order status updated');
                header('Location: order-view.php?id=' . $orderId);
                exit;
            }
        }
        
        if ($action === 'update_payment') {
            $paymentStatus = sanitize($_POST['payment_status'] ?? '');
            $validPaymentStatuses = ['pending', 'paid', 'failed', 'refunded'];
            
            if (in_array($paymentStatus, $validPaymentStatuses)) {
                query("UPDATE orders SET payment_status = ? WHERE id = ?", [$paymentStatus, $orderId]);
                setFlash('success', 'Payment status updated');
                header('Location: order-view.php?id=' . $orderId);
                exit;
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order #<?php echo htmlspecialchars($order['order_number']); ?> - Admin Panel</title>
    <link rel="stylesheet" href="../assets/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root { --primary: #1e3a5f; --primary-light: #2d4a6f; --success: #10b981; --danger: #ef4444; --warning: #f59e0b; --gray-100: #f3f4f6; --gray-200: #e5e7eb; --gray-500: #6b7280; --gray-700: #374151; --gray-900: #111827; }
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', system-ui, sans-serif; background: var(--gray-100); color: var(--gray-700); }
        .admin-wrapper { display: flex; min-height: 100vh; }
        .sidebar { width: 260px; background: var(--primary); color: #fff; position: fixed; height: 100vh; overflow-y: auto; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid rgba(255,255,255,0.1); }
        .sidebar-header h2 { font-size: 20px; font-weight: 700; }
        .sidebar-header span { font-size: 12px; opacity: 0.7; }
        .sidebar-menu { padding: 20px 0; }
        .menu-item { display: flex; align-items: center; padding: 14px 20px; color: rgba(255,255,255,0.8); text-decoration: none; transition: all 0.3s; gap: 12px; }
        .menu-item:hover, .menu-item.active { background: rgba(255,255,255,0.1); color: #fff; }
        .menu-item i { width: 20px; text-align: center; }
        .main-content { flex: 1; margin-left: 260px; padding: 30px; }
        .page-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px; flex-wrap: wrap; gap: 16px; }
        .page-header h1 { font-size: 24px; color: var(--gray-900); }
        .btn { padding: 10px 20px; border-radius: 8px; text-decoration: none; font-weight: 500; cursor: pointer; border: none; transition: all 0.3s; display: inline-flex; align-items: center; gap: 8px; }
        .btn-primary { background: var(--primary); color: #fff; }
        .btn-secondary { background: var(--gray-200); color: var(--gray-700); }
        .card { background: #fff; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.04); margin-bottom: 24px; }
        .card-header { padding: 20px 24px; border-bottom: 1px solid var(--gray-200); display: flex; justify-content: space-between; align-items: center; }
        .card-header h3 { font-size: 18px; color: var(--gray-900); }
        .card-body { padding: 24px; }
        .grid-3 { display: grid; grid-template-columns: repeat(3, 1fr); gap: 24px; }
        .grid-2 { display: grid; grid-template-columns: 2fr 1fr; gap: 24px; }
        .info-row { display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid var(--gray-100); }
        .info-row:last-child { border-bottom: none; }
        .info-label { color: var(--gray-500); }
        .info-value { font-weight: 500; text-align: right; }
        .badge { padding: 6px 12px; border-radius: 20px; font-size: 12px; font-weight: 600; }
        .badge-pending { background: #fef3c7; color: #d97706; }
        .badge-paid { background: #d1fae5; color: #059669; }
        .badge-processing { background: #dbeafe; color: #2563eb; }
        .badge-shipped { background: #e0e7ff; color: #4f46e5; }
        .badge-delivered { background: #d1fae5; color: #059669; }
        .badge-cancelled { background: #fee2e2; color: #dc2626; }
        .badge-failed { background: #fee2e2; color: #dc2626; }
        .badge-refunded { background: #fce7f3; color: #db2777; }
        .badge-success { background: #d1fae5; color: #059669; }
        .order-item { display: flex; gap: 16px; padding: 16px 0; border-bottom: 1px solid var(--gray-200); }
        .order-item:last-child { border-bottom: none; }
        .order-item img { width: 60px; height: 60px; object-fit: cover; border-radius: 8px; background: var(--gray-200); }
        .order-item-info { flex: 1; }
        .order-item-name { font-weight: 600; color: var(--gray-900); margin-bottom: 4px; }
        .order-item-price { color: var(--gray-500); }
        .order-item-total { text-align: right; font-weight: 600; }
        .totals-row { display: flex; justify-content: space-between; padding: 10px 0; }
        .totals-row.grand { font-size: 18px; font-weight: 700; color: var(--gray-900); border-top: 2px solid var(--gray-200); padding-top: 16px; margin-top: 8px; }
        .form-group { margin-bottom: 16px; }
        .form-group label { display: block; margin-bottom: 6px; font-weight: 500; }
        .form-group select { width: 100%; padding: 10px 14px; border: 1px solid var(--gray-200); border-radius: 8px; }
        .alert { padding: 12px 16px; border-radius: 8px; margin-bottom: 20px; }
        .alert-success { background: #d1fae5; color: #059669; }
        .alert-danger { background: #fee2e2; color: #dc2626; }
        @media (max-width: 1024px) { .grid-3, .grid-2 { grid-template-columns: 1fr; } }
        @media (max-width: 768px) {
            .sidebar { width: 100%; height: auto; position: relative; }
            .main-content { margin-left: 0; }
            .admin-wrapper { flex-direction: column; }
        }
    </style>
</head>
<body>
    <div class="admin-wrapper">
        <aside class="sidebar">
            <div class="sidebar-header">
                <h2><?php echo htmlspecialchars($settings['site_name']); ?></h2>
                <span>Admin Panel</span>
            </div>
            <nav class="sidebar-menu">
                <a href="dashboard.php" class="menu-item"><i class="fas fa-tachometer-alt"></i> Dashboard</a>
                <a href="products.php" class="menu-item"><i class="fas fa-box"></i> Products</a>
                <a href="categories.php" class="menu-item"><i class="fas fa-tags"></i> Categories</a>
                <a href="orders.php" class="menu-item active"><i class="fas fa-shopping-cart"></i> Orders</a>
                <a href="users.php" class="menu-item"><i class="fas fa-users"></i> Users</a>
                <a href="settings.php" class="menu-item"><i class="fas fa-cog"></i> Settings</a>
                <a href="logout.php" class="menu-item"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </nav>
        </aside>

        <main class="main-content">
            <div class="page-header">
                <h1>Order #<?php echo htmlspecialchars($order['order_number']); ?></h1>
                <div>
                    <a href="orders.php" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back to Orders</a>
                </div>
            </div>

            <?php echo displayFlash(); ?>

            <!-- Status Cards -->
            <div class="grid-3">
                <div class="card">
                    <div class="card-body">
                        <div class="info-row">
                            <span class="info-label">Order Status</span>
                            <span class="badge badge-<?php echo $order['status']; ?>"><?php echo ucfirst($order['status']); ?></span>
                        </div>
                        <div class="info-row">
                            <span class="info-label">Payment Status</span>
                            <span class="badge badge-<?php echo $order['payment_status']; ?>"><?php echo ucfirst($order['payment_status']); ?></span>
                        </div>
                        <div class="info-row">
                            <span class="info-label">Payment Method</span>
                            <span class="info-value"><?php echo htmlspecialchars($order['payment_method'] ?? 'N/A'); ?></span>
                        </div>
                    </div>
                </div>
                <div class="card">
                    <div class="card-body">
                        <div class="info-row">
                            <span class="info-label">Order Date</span>
                            <span class="info-value"><?php echo date('M d, Y H:i', strtotime($order['created_at'])); ?></span>
                        </div>
                        <div class="info-row">
                            <span class="info-label">Last Updated</span>
                            <span class="info-value"><?php echo date('M d, Y H:i', strtotime($order['updated_at'])); ?></span>
                        </div>
                    </div>
                </div>
                <div class="card">
                    <div class="card-body">
                        <div class="info-row">
                            <span class="info-label">Order Total</span>
                            <span class="info-value" style="font-size: 20px; color: var(--success);">
                                <?php echo formatPrice($order['total']); ?>
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <div class="grid-2">
                <!-- Order Items -->
                <div>
                    <div class="card">
                        <div class="card-header">
                            <h3>Order Items</h3>
                        </div>
                        <div class="card-body">
                            <?php foreach ($orderItems as $item): ?>
                            <div class="order-item">
                                <?php if ($item['product_image']): ?>
                                <img src="../uploads/products/<?php echo htmlspecialchars($item['product_image']); ?>" alt="">
                                <?php else: ?>
                                <div style="width: 60px; height: 60px; background: var(--gray-200); border-radius: 8px; display: flex; align-items: center; justify-content: center; color: var(--gray-500);"><i class="fas fa-image"></i></div>
                                <?php endif; ?>
                                <div class="order-item-info">
                                    <div class="order-item-name"><?php echo htmlspecialchars($item['product_name']); ?></div>
                                    <div class="order-item-price"><?php echo formatPrice($item['price']); ?> x <?php echo $item['quantity']; ?></div>
                                </div>
                                <div class="order-item-total"><?php echo formatPrice($item['total']); ?></div>
                            </div>
                            <?php endforeach; ?>

                            <div style="margin-top: 20px; padding-top: 20px; border-top: 1px solid var(--gray-200);">
                                <div class="totals-row">
                                    <span>Subtotal</span>
                                    <span><?php echo formatPrice($order['subtotal']); ?></span>
                                </div>
                                <div class="totals-row">
                                    <span>Shipping</span>
                                    <span><?php echo formatPrice($order['shipping_cost']); ?></span>
                                </div>
                                <?php if ($order['tax'] > 0): ?>
                                <div class="totals-row">
                                    <span>Tax</span>
                                    <span><?php echo formatPrice($order['tax']); ?></span>
                                </div>
                                <?php endif; ?>
                                <?php if ($order['discount'] > 0): ?>
                                <div class="totals-row">
                                    <span>Discount</span>
                                    <span>-<?php echo formatPrice($order['discount']); ?></span>
                                </div>
                                <?php endif; ?>
                                <div class="totals-row grand">
                                    <span>Total</span>
                                    <span><?php echo formatPrice($order['total']); ?></span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Shipping Info -->
                    <div class="card">
                        <div class="card-header">
                            <h3>Shipping Information</h3>
                        </div>
                        <div class="card-body">
                            <div class="info-row">
                                <span class="info-label">Name</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['shipping_name']); ?></span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">Email</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['shipping_email']); ?></span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">Phone</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['shipping_phone']); ?></span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">Address</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['shipping_address']); ?></span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">City</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['shipping_city']); ?></span>
                            </div>
                            <?php if ($order['shipping_postal_code']): ?>
                            <div class="info-row">
                                <span class="info-label">Postal Code</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['shipping_postal_code']); ?></span>
                            </div>
                            <?php endif; ?>
                            <div class="info-row">
                                <span class="info-label">Country</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['shipping_country']); ?></span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Side Panel -->
                <div>
                    <!-- Update Status -->
                    <div class="card">
                        <div class="card-header">
                            <h3>Update Order Status</h3>
                        </div>
                        <div class="card-body">
                            <form method="POST">
                                <?php echo csrfField(); ?>
                                <input type="hidden" name="action" value="update_status">
                                <div class="form-group">
                                    <label>Order Status</label>
                                    <select name="status">
                                        <option value="pending" <?php echo $order['status'] === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                        <option value="paid" <?php echo $order['status'] === 'paid' ? 'selected' : ''; ?>>Paid</option>
                                        <option value="processing" <?php echo $order['status'] === 'processing' ? 'selected' : ''; ?>>Processing</option>
                                        <option value="shipped" <?php echo $order['status'] === 'shipped' ? 'selected' : ''; ?>>Shipped</option>
                                        <option value="delivered" <?php echo $order['status'] === 'delivered' ? 'selected' : ''; ?>>Delivered</option>
                                        <option value="cancelled" <?php echo $order['status'] === 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                                        <option value="refunded" <?php echo $order['status'] === 'refunded' ? 'selected' : ''; ?>>Refunded</option>
                                    </select>
                                </div>
                                <button type="submit" class="btn btn-primary" style="width: 100%;">Update Status</button>
                            </form>
                        </div>
                    </div>

                    <!-- Payment Status -->
                    <div class="card">
                        <div class="card-header">
                            <h3>Payment Status</h3>
                        </div>
                        <div class="card-body">
                            <form method="POST">
                                <?php echo csrfField(); ?>
                                <input type="hidden" name="action" value="update_payment">
                                <div class="form-group">
                                    <label>Payment Status</label>
                                    <select name="payment_status">
                                        <option value="pending" <?php echo $order['payment_status'] === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                        <option value="paid" <?php echo $order['payment_status'] === 'paid' ? 'selected' : ''; ?>>Paid</option>
                                        <option value="failed" <?php echo $order['payment_status'] === 'failed' ? 'selected' : ''; ?>>Failed</option>
                                        <option value="refunded" <?php echo $order['payment_status'] === 'refunded' ? 'selected' : ''; ?>>Refunded</option>
                                    </select>
                                </div>
                                <button type="submit" class="btn btn-primary" style="width: 100%;">Update Payment</button>
                            </form>
                        </div>
                    </div>

                    <!-- Payment Details -->
                    <?php if ($payment): ?>
                    <div class="card">
                        <div class="card-header">
                            <h3>Payment Details</h3>
                        </div>
                        <div class="card-body">
                            <div class="info-row">
                                <span class="info-label">Transaction ID</span>
                                <span class="info-value" style="font-size: 12px;"><?php echo htmlspecialchars($payment['transaction_id'] ?? 'N/A'); ?></span>
                            </div>
                            <?php if ($payment['val_id']): ?>
                            <div class="info-row">
                                <span class="info-label">Validation ID</span>
                                <span class="info-value" style="font-size: 12px;"><?php echo htmlspecialchars($payment['val_id']); ?></span>
                            </div>
                            <?php endif; ?>
                            <div class="info-row">
                                <span class="info-label">Amount</span>
                                <span class="info-value"><?php echo formatPrice($payment['amount']); ?></span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">Method</span>
                                <span class="info-value"><?php echo htmlspecialchars($payment['payment_method']); ?></span>
                            </div>
                            <?php if ($payment['card_type']): ?>
                            <div class="info-row">
                                <span class="info-label">Card Type</span>
                                <span class="info-value"><?php echo htmlspecialchars($payment['card_type']); ?></span>
                            </div>
                            <?php endif; ?>
                            <div class="info-row">
                                <span class="info-label">Status</span>
                                <span class="badge badge-<?php echo $payment['status']; ?>"><?php echo ucfirst($payment['status']); ?></span>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>

                    <!-- Customer Info -->
                    <?php if ($order['user_id']): ?>
                    <div class="card">
                        <div class="card-header">
                            <h3>Customer Account</h3>
                        </div>
                        <div class="card-body">
                            <div class="info-row">
                                <span class="info-label">Name</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['user_name']); ?></span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">Email</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['user_email']); ?></span>
                            </div>
                            <a href="users.php?view=<?php echo $order['user_id']; ?>" class="btn btn-secondary" style="width: 100%; margin-top: 12px;">View Customer</a>
                        </div>
                    </div>
                    <?php endif; ?>

                    <!-- Notes -->
                    <?php if ($order['notes']): ?>
                    <div class="card">
                        <div class="card-header">
                            <h3>Order Notes</h3>
                        </div>
                        <div class="card-body">
                            <p><?php echo nl2br(htmlspecialchars($order['notes'])); ?></p>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</body>
</html>
