<?php
/**
 * Admin Products Management
 */
define('ECOMMERCE_ACCESS', true);
require_once '../config/db.php';
require_once '../config/auth.php';

requireAdmin();

$settings = getSettings();
$admin = getCurrentAdmin();

// Handle delete
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    if (!verifyCSRFToken($_GET['token'] ?? '')) {
        setFlash('error', 'Invalid request');
    } else {
        $productId = (int)$_GET['delete'];
        query("DELETE FROM products WHERE id = ?", [$productId]);
        setFlash('success', 'Product deleted successfully');
    }
    header('Location: products.php');
    exit;
}

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 10;
$offset = ($page - 1) * $limit;

// Search and filter
$search = sanitize($_GET['search'] ?? '');
$category = intval($_GET['category'] ?? 0);
$status = sanitize($_GET['status'] ?? '');

$where = "1=1";
$params = [];

if ($search) {
    $where .= " AND (p.name LIKE ? OR p.sku LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}
if ($category) {
    $where .= " AND p.category_id = ?";
    $params[] = $category;
}
if ($status) {
    $where .= " AND p.status = ?";
    $params[] = $status;
}

// Get total count
$totalCount = fetchOne("SELECT COUNT(*) as count FROM products p WHERE $where", $params)['count'];
$totalPages = ceil($totalCount / $limit);

// Get products
$products = fetchAll("
    SELECT p.*, c.name as category_name 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    WHERE $where 
    ORDER BY p.created_at DESC 
    LIMIT $limit OFFSET $offset
", $params);

// Get categories for filter
$categories = fetchAll("SELECT * FROM categories ORDER BY name");
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products - Admin Panel</title>
    <link rel="stylesheet" href="../assets/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #1e3a5f;
            --primary-light: #2d4a6f;
            --success: #10b981;
            --danger: #ef4444;
            --warning: #f59e0b;
            --gray-100: #f3f4f6;
            --gray-200: #e5e7eb;
            --gray-500: #6b7280;
            --gray-700: #374151;
            --gray-900: #111827;
        }
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', system-ui, sans-serif; background: var(--gray-100); color: var(--gray-700); }
        .admin-wrapper { display: flex; min-height: 100vh; }
        .sidebar { width: 260px; background: var(--primary); color: #fff; position: fixed; height: 100vh; overflow-y: auto; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid rgba(255,255,255,0.1); }
        .sidebar-header h2 { font-size: 20px; font-weight: 700; }
        .sidebar-header span { font-size: 12px; opacity: 0.7; }
        .sidebar-menu { padding: 20px 0; }
        .menu-item { display: flex; align-items: center; padding: 14px 20px; color: rgba(255,255,255,0.8); text-decoration: none; transition: all 0.3s; gap: 12px; }
        .menu-item:hover, .menu-item.active { background: rgba(255,255,255,0.1); color: #fff; }
        .menu-item i { width: 20px; text-align: center; }
        .main-content { flex: 1; margin-left: 260px; padding: 30px; }
        .page-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px; }
        .page-header h1 { font-size: 28px; color: var(--gray-900); }
        .btn { padding: 10px 20px; border-radius: 8px; text-decoration: none; font-weight: 500; cursor: pointer; border: none; transition: all 0.3s; display: inline-flex; align-items: center; gap: 8px; }
        .btn-primary { background: var(--primary); color: #fff; }
        .btn-primary:hover { background: var(--primary-light); }
        .btn-danger { background: var(--danger); color: #fff; }
        .btn-sm { padding: 6px 12px; font-size: 13px; }
        .card { background: #fff; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.04); margin-bottom: 24px; }
        .card-header { padding: 20px 24px; border-bottom: 1px solid var(--gray-200); }
        .card-body { padding: 24px; }
        .filters { display: flex; gap: 16px; flex-wrap: wrap; margin-bottom: 24px; }
        .filters input, .filters select { padding: 10px 14px; border: 1px solid var(--gray-200); border-radius: 8px; font-size: 14px; }
        .filters input { min-width: 250px; }
        .table { width: 100%; border-collapse: collapse; }
        .table th, .table td { padding: 14px 16px; text-align: left; border-bottom: 1px solid var(--gray-200); }
        .table th { font-weight: 600; color: var(--gray-500); font-size: 13px; text-transform: uppercase; background: var(--gray-100); }
        .product-img { width: 50px; height: 50px; object-fit: cover; border-radius: 8px; background: var(--gray-200); }
        .badge { padding: 4px 10px; border-radius: 20px; font-size: 12px; font-weight: 600; }
        .badge-active { background: #d1fae5; color: #059669; }
        .badge-inactive { background: #fee2e2; color: #dc2626; }
        .badge-draft { background: #fef3c7; color: #d97706; }
        .actions { display: flex; gap: 8px; }
        .pagination { display: flex; justify-content: center; gap: 8px; margin-top: 24px; }
        .pagination a, .pagination span { padding: 8px 14px; border-radius: 6px; text-decoration: none; color: var(--gray-700); background: #fff; border: 1px solid var(--gray-200); }
        .pagination a:hover { background: var(--primary); color: #fff; }
        .pagination .active { background: var(--primary); color: #fff; }
        .alert { padding: 14px 20px; border-radius: 8px; margin-bottom: 20px; }
        .alert-success { background: #d1fae5; color: #059669; }
        .alert-danger { background: #fee2e2; color: #dc2626; }
        .stock-low { color: var(--danger); font-weight: 600; }
        @media (max-width: 768px) {
            .sidebar { width: 100%; height: auto; position: relative; }
            .main-content { margin-left: 0; }
            .admin-wrapper { flex-direction: column; }
            .filters { flex-direction: column; }
        }
    </style>
</head>
<body>
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <h2><?php echo htmlspecialchars($settings['site_name']); ?></h2>
                <span>Admin Panel</span>
            </div>
            <nav class="sidebar-menu">
                <a href="dashboard.php" class="menu-item"><i class="fas fa-tachometer-alt"></i> Dashboard</a>
                <a href="products.php" class="menu-item active"><i class="fas fa-box"></i> Products</a>
                <a href="categories.php" class="menu-item"><i class="fas fa-tags"></i> Categories</a>
                <a href="orders.php" class="menu-item"><i class="fas fa-shopping-cart"></i> Orders</a>
                <a href="users.php" class="menu-item"><i class="fas fa-users"></i> Users</a>
                <a href="settings.php" class="menu-item"><i class="fas fa-cog"></i> Settings</a>
                <a href="logout.php" class="menu-item"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="main-content">
            <div class="page-header">
                <h1>Products</h1>
                <a href="add-product.php" class="btn btn-primary"><i class="fas fa-plus"></i> Add Product</a>
            </div>

            <?php echo displayFlash(); ?>

            <!-- Filters -->
            <div class="card">
                <div class="card-body">
                    <form method="GET" class="filters">
                        <input type="text" name="search" placeholder="Search products..." value="<?php echo htmlspecialchars($search); ?>">
                        <select name="category">
                            <option value="">All Categories</option>
                            <?php foreach ($categories as $cat): ?>
                            <option value="<?php echo $cat['id']; ?>" <?php echo $category == $cat['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($cat['name']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                        <select name="status">
                            <option value="">All Status</option>
                            <option value="active" <?php echo $status === 'active' ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo $status === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                            <option value="draft" <?php echo $status === 'draft' ? 'selected' : ''; ?>>Draft</option>
                        </select>
                        <button type="submit" class="btn btn-primary btn-sm">Filter</button>
                    </form>
                </div>
            </div>

            <!-- Products Table -->
            <div class="card">
                <div class="card-body">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Image</th>
                                <th>Name</th>
                                <th>Category</th>
                                <th>Price</th>
                                <th>Stock</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($products)): ?>
                            <tr>
                                <td colspan="7" style="text-align: center; padding: 40px; color: var(--gray-500);">
                                    No products found. <a href="add-product.php">Add your first product</a>
                                </td>
                            </tr>
                            <?php else: ?>
                            <?php foreach ($products as $product): ?>
                            <tr>
                                <td>
                                    <?php if ($product['image']): ?>
                                    <img src="../uploads/products/<?php echo htmlspecialchars($product['image']); ?>" alt="" class="product-img">
                                    <?php else: ?>
                                    <div class="product-img" style="display: flex; align-items: center; justify-content: center; color: var(--gray-500);"><i class="fas fa-image"></i></div>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <strong><?php echo htmlspecialchars($product['name']); ?></strong>
                                    <?php if ($product['sku']): ?>
                                    <br><small style="color: var(--gray-500);">SKU: <?php echo htmlspecialchars($product['sku']); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo htmlspecialchars($product['category_name'] ?? 'Uncategorized'); ?></td>
                                <td>
                                    <?php if ($product['sale_price']): ?>
                                    <span style="text-decoration: line-through; color: var(--gray-500);"><?php echo formatPrice($product['price']); ?></span><br>
                                    <strong><?php echo formatPrice($product['sale_price']); ?></strong>
                                    <?php else: ?>
                                    <?php echo formatPrice($product['price']); ?>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="<?php echo $product['stock'] <= 5 ? 'stock-low' : ''; ?>">
                                        <?php echo $product['stock']; ?>
                                    </span>
                                </td>
                                <td><span class="badge badge-<?php echo $product['status']; ?>"><?php echo ucfirst($product['status']); ?></span></td>
                                <td class="actions">
                                    <a href="edit-product.php?id=<?php echo $product['id']; ?>" class="btn btn-primary btn-sm" title="Edit"><i class="fas fa-edit"></i></a>
                                    <a href="products.php?delete=<?php echo $product['id']; ?>&token=<?php echo generateCSRFToken(); ?>" 
                                       class="btn btn-danger btn-sm" 
                                       onclick="return confirm('Are you sure you want to delete this product?')"
                                       title="Delete"><i class="fas fa-trash"></i></a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>

                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                    <div class="pagination">
                        <?php if ($page > 1): ?>
                        <a href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&category=<?php echo $category; ?>&status=<?php echo urlencode($status); ?>">&laquo; Prev</a>
                        <?php endif; ?>
                        
                        <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                        <?php if ($i == $page): ?>
                        <span class="active"><?php echo $i; ?></span>
                        <?php else: ?>
                        <a href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&category=<?php echo $category; ?>&status=<?php echo urlencode($status); ?>"><?php echo $i; ?></a>
                        <?php endif; ?>
                        <?php endfor; ?>
                        
                        <?php if ($page < $totalPages): ?>
                        <a href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&category=<?php echo $category; ?>&status=<?php echo urlencode($status); ?>">Next &raquo;</a>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</body>
</html>
