<?php
/**
 * Payment Gateway Configuration
 * E-Commerce Application - SSLCommerz Integration
 */

// Prevent direct access
if (!defined('ECOMMERCE_ACCESS')) {
    die('Direct access not permitted');
}

class SSLCommerz {
    private $storeId;
    private $storePassword;
    private $isSandbox;
    private $apiUrl;
    private $validateUrl;
    
    public function __construct() {
        $settings = getSettings();
        $this->storeId = $settings['ssl_store_id'] ?? '';
        $this->storePassword = $settings['ssl_store_password'] ?? '';
        $this->isSandbox = $settings['ssl_sandbox'] ?? true;
        
        if ($this->isSandbox) {
            $this->apiUrl = 'https://sandbox.sslcommerz.com/gwprocess/v4/api.php';
            $this->validateUrl = 'https://sandbox.sslcommerz.com/validator/api/validationserverAPI.php';
        } else {
            $this->apiUrl = 'https://securepay.sslcommerz.com/gwprocess/v4/api.php';
            $this->validateUrl = 'https://securepay.sslcommerz.com/validator/api/validationserverAPI.php';
        }
    }
    
    public function initiatePayment($order, $user, $callbackUrls) {
        $postData = [
            'store_id' => $this->storeId,
            'store_passwd' => $this->storePassword,
            'total_amount' => $order['total'],
            'currency' => 'BDT',
            'tran_id' => $order['order_number'],
            'success_url' => $callbackUrls['success'],
            'fail_url' => $callbackUrls['fail'],
            'cancel_url' => $callbackUrls['cancel'],
            'ipn_url' => $callbackUrls['ipn'],
            
            // Customer information
            'cus_name' => $order['shipping_name'],
            'cus_email' => $order['shipping_email'],
            'cus_add1' => $order['shipping_address'],
            'cus_city' => $order['shipping_city'],
            'cus_postcode' => $order['shipping_postal_code'] ?? '',
            'cus_country' => $order['shipping_country'],
            'cus_phone' => $order['shipping_phone'],
            
            // Shipping information
            'shipping_method' => 'NO',
            'num_of_item' => 1,
            'product_name' => 'Order #' . $order['order_number'],
            'product_category' => 'E-Commerce',
            'product_profile' => 'general',
            
            // Optional
            'value_a' => $order['id'],
            'value_b' => $user['id'] ?? 'guest',
        ];
        
        $response = $this->sendRequest($this->apiUrl, $postData);
        
        if ($response && isset($response['status']) && $response['status'] === 'SUCCESS') {
            return [
                'success' => true,
                'gateway_url' => $response['GatewayPageURL'],
                'session_key' => $response['sessionkey']
            ];
        }
        
        return [
            'success' => false,
            'message' => $response['failedreason'] ?? 'Payment initialization failed'
        ];
    }
    
    public function validateTransaction($valId) {
        $postData = [
            'store_id' => $this->storeId,
            'store_passwd' => $this->storePassword,
            'val_id' => $valId,
            'format' => 'json'
        ];
        
        $response = $this->sendRequest($this->validateUrl, $postData);
        
        if ($response && isset($response['status']) && $response['status'] === 'VALID') {
            return [
                'success' => true,
                'data' => $response
            ];
        }
        
        return [
            'success' => false,
            'message' => 'Transaction validation failed',
            'data' => $response
        ];
    }
    
    public function verifyPayment($postData) {
        // Verify the payment data received from SSLCommerz
        if (!isset($postData['val_id']) || !isset($postData['tran_id'])) {
            return ['success' => false, 'message' => 'Invalid payment data'];
        }
        
        // Validate transaction
        $validation = $this->validateTransaction($postData['val_id']);
        
        if (!$validation['success']) {
            return $validation;
        }
        
        $validData = $validation['data'];
        
        // Verify store credentials
        if ($validData['store_id'] !== $this->storeId) {
            return ['success' => false, 'message' => 'Store ID mismatch'];
        }
        
        return [
            'success' => true,
            'transaction_id' => $validData['tran_id'],
            'val_id' => $validData['val_id'],
            'amount' => $validData['amount'],
            'currency' => $validData['currency'],
            'card_type' => $validData['card_type'] ?? null,
            'card_no' => $validData['card_no'] ?? null,
            'bank_tran_id' => $validData['bank_tran_id'] ?? null,
            'status' => $validData['status'],
            'risk_level' => $validData['risk_level'] ?? null,
            'risk_title' => $validData['risk_title'] ?? null
        ];
    }
    
    private function sendRequest($url, $data) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return null;
        }
        
        return json_decode($response, true);
    }
}

// Helper function to get SSLCommerz instance
function getSSLCommerz() {
    return new SSLCommerz();
}

// Get base URL
function getBaseUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $script = dirname($_SERVER['SCRIPT_NAME']);
    $basePath = rtrim(dirname($script), '/');
    return $protocol . '://' . $host . $basePath;
}
