<?php
/**
 * Shopping Cart
 */
define('ECOMMERCE_ACCESS', true);
require_once '../config/db.php';
require_once '../config/auth.php';

$settings = getSettings();
$user = getCurrentUser();
$sessionId = session_id();

// Handle add to cart
if (isset($_GET['add']) && is_numeric($_GET['add'])) {
    $productId = (int)$_GET['add'];
    $qty = max(1, intval($_GET['qty'] ?? 1));
    
    $product = fetchOne("SELECT * FROM products WHERE id = ? AND status = 'active'", [$productId]);
    if ($product && $product['stock'] >= $qty) {
        // Check if already in cart
        if ($user) {
            $existing = fetchOne("SELECT * FROM cart WHERE user_id = ? AND product_id = ?", [$user['id'], $productId]);
        } else {
            $existing = fetchOne("SELECT * FROM cart WHERE session_id = ? AND product_id = ?", [$sessionId, $productId]);
        }
        
        if ($existing) {
            $newQty = min($product['stock'], $existing['quantity'] + $qty);
            query("UPDATE cart SET quantity = ? WHERE id = ?", [$newQty, $existing['id']]);
        } else {
            if ($user) {
                query("INSERT INTO cart (user_id, product_id, quantity) VALUES (?, ?, ?)", [$user['id'], $productId, $qty]);
            } else {
                query("INSERT INTO cart (session_id, product_id, quantity) VALUES (?, ?, ?)", [$sessionId, $productId, $qty]);
            }
        }
        setFlash('success', 'Product added to cart!');
    }
    header('Location: cart.php');
    exit;
}

// Handle remove from cart
if (isset($_GET['remove']) && is_numeric($_GET['remove'])) {
    $cartId = (int)$_GET['remove'];
    if ($user) {
        query("DELETE FROM cart WHERE id = ? AND user_id = ?", [$cartId, $user['id']]);
    } else {
        query("DELETE FROM cart WHERE id = ? AND session_id = ?", [$cartId, $sessionId]);
    }
    setFlash('success', 'Item removed from cart');
    header('Location: cart.php');
    exit;
}

// Handle quantity update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_cart'])) {
    if (verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        foreach ($_POST['qty'] as $cartId => $qty) {
            $qty = max(1, intval($qty));
            if ($user) {
                query("UPDATE cart SET quantity = ? WHERE id = ? AND user_id = ?", [$qty, $cartId, $user['id']]);
            } else {
                query("UPDATE cart SET quantity = ? WHERE id = ? AND session_id = ?", [$qty, $cartId, $sessionId]);
            }
        }
        setFlash('success', 'Cart updated');
    }
    header('Location: cart.php');
    exit;
}

// Get cart items
if ($user) {
    $cartItems = fetchAll("
        SELECT c.*, p.name, p.price, p.sale_price, p.image, p.stock, p.slug 
        FROM cart c 
        JOIN products p ON c.product_id = p.id 
        WHERE c.user_id = ? AND p.status = 'active'
    ", [$user['id']]);
} else {
    $cartItems = fetchAll("
        SELECT c.*, p.name, p.price, p.sale_price, p.image, p.stock, p.slug 
        FROM cart c 
        JOIN products p ON c.product_id = p.id 
        WHERE c.session_id = ? AND p.status = 'active'
    ", [$sessionId]);
}

// Calculate totals
$subtotal = 0;
foreach ($cartItems as $item) {
    $price = $item['sale_price'] ?? $item['price'];
    $subtotal += $price * $item['quantity'];
}
$shipping = $subtotal > 0 ? 60 : 0; // Fixed shipping
$total = $subtotal + $shipping;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cart - <?php echo htmlspecialchars($settings['site_name']); ?></title>
    <link rel="stylesheet" href="../assets/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root { --primary: #0ea5e9; --primary-dark: #0284c7; --success: #10b981; --danger: #ef4444; --gray-100: #f1f5f9; --gray-200: #e2e8f0; --gray-500: #64748b; --gray-700: #334155; --gray-900: #0f172a; }
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', system-ui, sans-serif; background: var(--gray-100); color: var(--gray-700); }
        .header { background: #fff; padding: 16px 0; box-shadow: 0 1px 3px rgba(0,0,0,0.1); position: sticky; top: 0; z-index: 100; }
        .header-content { max-width: 1200px; margin: 0 auto; padding: 0 20px; display: flex; justify-content: space-between; align-items: center; }
        .logo { font-size: 24px; font-weight: 700; color: var(--primary); text-decoration: none; }
        .nav { display: flex; gap: 24px; align-items: center; }
        .nav a { color: var(--gray-700); text-decoration: none; font-weight: 500; transition: color 0.3s; }
        .nav a:hover { color: var(--primary); }
        .container { max-width: 1200px; margin: 0 auto; padding: 30px 20px; }
        .page-title { font-size: 28px; color: var(--gray-900); margin-bottom: 30px; }
        .cart-layout { display: grid; grid-template-columns: 1fr 380px; gap: 30px; }
        .cart-items { background: #fff; border-radius: 12px; padding: 24px; }
        .cart-item { display: grid; grid-template-columns: 100px 1fr auto auto; gap: 20px; align-items: center; padding: 20px 0; border-bottom: 1px solid var(--gray-100); }
        .cart-item:last-child { border-bottom: none; }
        .cart-item img { width: 100px; height: 100px; object-fit: cover; border-radius: 8px; background: var(--gray-100); }
        .item-details h3 { font-size: 16px; color: var(--gray-900); margin-bottom: 4px; }
        .item-details h3 a { color: inherit; text-decoration: none; }
        .item-details .price { color: var(--primary); font-weight: 600; }
        .item-details .original-price { color: var(--gray-500); text-decoration: line-through; font-size: 14px; margin-left: 8px; }
        .quantity-input { display: flex; align-items: center; border: 1px solid var(--gray-200); border-radius: 6px; overflow: hidden; }
        .quantity-input button { width: 36px; height: 36px; border: none; background: var(--gray-100); cursor: pointer; }
        .quantity-input button:hover { background: var(--gray-200); }
        .quantity-input input { width: 50px; height: 36px; border: none; text-align: center; font-weight: 500; }
        .item-total { font-size: 18px; font-weight: 700; color: var(--gray-900); min-width: 100px; text-align: right; }
        .remove-btn { color: var(--danger); background: none; border: none; cursor: pointer; padding: 8px; font-size: 18px; }
        .cart-summary { background: #fff; border-radius: 12px; padding: 24px; height: fit-content; position: sticky; top: 100px; }
        .summary-title { font-size: 20px; color: var(--gray-900); margin-bottom: 20px; padding-bottom: 16px; border-bottom: 1px solid var(--gray-200); }
        .summary-row { display: flex; justify-content: space-between; margin-bottom: 12px; }
        .summary-row.total { font-size: 20px; font-weight: 700; color: var(--gray-900); padding-top: 16px; border-top: 2px solid var(--gray-200); margin-top: 16px; }
        .btn { padding: 14px 28px; border-radius: 10px; text-decoration: none; font-weight: 600; cursor: pointer; border: none; transition: all 0.3s; display: inline-flex; align-items: center; justify-content: center; gap: 10px; width: 100%; font-size: 16px; }
        .btn-primary { background: var(--primary); color: #fff; }
        .btn-primary:hover { background: var(--primary-dark); }
        .btn-secondary { background: var(--gray-100); color: var(--gray-700); margin-top: 12px; }
        .empty-cart { text-align: center; padding: 60px 20px; background: #fff; border-radius: 12px; }
        .empty-cart i { font-size: 80px; color: var(--gray-200); margin-bottom: 24px; }
        .empty-cart h2 { color: var(--gray-900); margin-bottom: 12px; }
        .empty-cart p { color: var(--gray-500); margin-bottom: 24px; }
        .alert { padding: 14px 20px; border-radius: 8px; margin-bottom: 20px; }
        .alert-success { background: #d1fae5; color: #059669; }
        @media (max-width: 768px) {
            .cart-layout { grid-template-columns: 1fr; }
            .cart-item { grid-template-columns: 80px 1fr; }
            .item-total { grid-column: span 2; text-align: left; margin-top: 10px; }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <a href="../index.php" class="logo"><?php echo htmlspecialchars($settings['site_name']); ?></a>
            <nav class="nav">
                <a href="products.php">Shop</a>
                <a href="cart.php">Cart (<?php echo count($cartItems); ?>)</a>
                <?php if ($user): ?>
                <a href="dashboard.php">My Account</a>
                <a href="logout.php">Logout</a>
                <?php else: ?>
                <a href="login.php">Login</a>
                <?php endif; ?>
            </nav>
        </div>
    </header>

    <div class="container">
        <h1 class="page-title">Shopping Cart</h1>
        
        <?php echo displayFlash(); ?>

        <?php if (empty($cartItems)): ?>
        <div class="empty-cart">
            <i class="fas fa-shopping-cart"></i>
            <h2>Your cart is empty</h2>
            <p>Looks like you haven't added any products to your cart yet.</p>
            <a href="products.php" class="btn btn-primary" style="width: auto; display: inline-flex;">
                <i class="fas fa-store"></i> Continue Shopping
            </a>
        </div>
        <?php else: ?>
        <form method="POST">
            <?php echo csrfField(); ?>
            <div class="cart-layout">
                <div class="cart-items">
                    <?php foreach ($cartItems as $item): ?>
                    <?php $price = $item['sale_price'] ?? $item['price']; ?>
                    <div class="cart-item">
                        <?php if ($item['image']): ?>
                        <img src="../uploads/products/<?php echo htmlspecialchars($item['image']); ?>" 
                             alt="<?php echo htmlspecialchars($item['name']); ?>">
                        <?php else: ?>
                        <div style="width: 100px; height: 100px; background: var(--gray-100); border-radius: 8px; display: flex; align-items: center; justify-content: center;">
                            <i class="fas fa-image" style="color: var(--gray-300); font-size: 24px;"></i>
                        </div>
                        <?php endif; ?>
                        
                        <div class="item-details">
                            <h3><a href="product-view.php?slug=<?php echo htmlspecialchars($item['slug']); ?>"><?php echo htmlspecialchars($item['name']); ?></a></h3>
                            <div>
                                <span class="price"><?php echo formatPrice($price); ?></span>
                                <?php if ($item['sale_price']): ?>
                                <span class="original-price"><?php echo formatPrice($item['price']); ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="quantity-input">
                            <button type="button" onclick="changeQty(<?php echo $item['id']; ?>, -1)">-</button>
                            <input type="number" name="qty[<?php echo $item['id']; ?>]" 
                                   id="qty-<?php echo $item['id']; ?>"
                                   value="<?php echo $item['quantity']; ?>" 
                                   min="1" max="<?php echo $item['stock']; ?>">
                            <button type="button" onclick="changeQty(<?php echo $item['id']; ?>, 1)">+</button>
                        </div>
                        
                        <div class="item-total"><?php echo formatPrice($price * $item['quantity']); ?></div>
                        
                        <a href="cart.php?remove=<?php echo $item['id']; ?>" class="remove-btn" title="Remove">
                            <i class="fas fa-trash"></i>
                        </a>
                    </div>
                    <?php endforeach; ?>
                    
                    <div style="margin-top: 20px;">
                        <button type="submit" name="update_cart" class="btn btn-secondary" style="width: auto;">
                            <i class="fas fa-sync"></i> Update Cart
                        </button>
                    </div>
                </div>

                <div class="cart-summary">
                    <h2 class="summary-title">Order Summary</h2>
                    <div class="summary-row">
                        <span>Subtotal (<?php echo count($cartItems); ?> items)</span>
                        <span><?php echo formatPrice($subtotal); ?></span>
                    </div>
                    <div class="summary-row">
                        <span>Shipping</span>
                        <span><?php echo formatPrice($shipping); ?></span>
                    </div>
                    <div class="summary-row total">
                        <span>Total</span>
                        <span><?php echo formatPrice($total); ?></span>
                    </div>
                    <a href="checkout.php" class="btn btn-primary" style="margin-top: 20px;">
                        <i class="fas fa-lock"></i> Proceed to Checkout
                    </a>
                    <a href="products.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> Continue Shopping
                    </a>
                </div>
            </div>
        </form>
        <?php endif; ?>
    </div>

    <script>
        function changeQty(id, delta) {
            const input = document.getElementById('qty-' + id);
            const newVal = parseInt(input.value) + delta;
            if (newVal >= 1 && newVal <= parseInt(input.max)) {
                input.value = newVal;
            }
        }
    </script>
</body>
</html>
