<?php
/**
 * Single Product View
 */
define('ECOMMERCE_ACCESS', true);
require_once '../config/db.php';
require_once '../config/auth.php';

$settings = getSettings();
$user = getCurrentUser();

// Get product by slug
$slug = sanitize($_GET['slug'] ?? '');
if (!$slug) {
    header('Location: products.php');
    exit;
}

$product = fetchOne("
    SELECT p.*, c.name as category_name 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    WHERE p.slug = ? AND p.status = 'active'
", [$slug]);

if (!$product) {
    header('Location: products.php');
    exit;
}

// Increment view count
query("UPDATE products SET views = views + 1 WHERE id = ?", [$product['id']]);

// Related products
$relatedProducts = fetchAll("
    SELECT * FROM products 
    WHERE category_id = ? AND id != ? AND status = 'active' 
    ORDER BY RAND() 
    LIMIT 4
", [$product['category_id'], $product['id']]);

// Cart count
$cartCount = 0;
if ($user) {
    $cartCount = fetchOne("SELECT COUNT(*) as count FROM cart WHERE user_id = ?", [$user['id']])['count'];
} else {
    $cartCount = fetchOne("SELECT COUNT(*) as count FROM cart WHERE session_id = ?", [session_id()])['count'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($product['name']); ?> - <?php echo htmlspecialchars($settings['site_name']); ?></title>
    <link rel="stylesheet" href="../assets/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root { --primary: #0ea5e9; --primary-dark: #0284c7; --success: #10b981; --danger: #ef4444; --gray-100: #f1f5f9; --gray-200: #e2e8f0; --gray-500: #64748b; --gray-700: #334155; --gray-900: #0f172a; }
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', system-ui, sans-serif; background: var(--gray-100); color: var(--gray-700); }
        .header { background: #fff; padding: 16px 0; box-shadow: 0 1px 3px rgba(0,0,0,0.1); position: sticky; top: 0; z-index: 100; }
        .header-content { max-width: 1200px; margin: 0 auto; padding: 0 20px; display: flex; justify-content: space-between; align-items: center; }
        .logo { font-size: 24px; font-weight: 700; color: var(--primary); text-decoration: none; }
        .nav { display: flex; gap: 24px; align-items: center; }
        .nav a { color: var(--gray-700); text-decoration: none; font-weight: 500; transition: color 0.3s; }
        .nav a:hover { color: var(--primary); }
        .nav .cart-link { position: relative; }
        .cart-badge { position: absolute; top: -8px; right: -12px; background: var(--primary); color: #fff; font-size: 11px; padding: 2px 6px; border-radius: 10px; }
        .container { max-width: 1200px; margin: 0 auto; padding: 30px 20px; }
        .breadcrumb { display: flex; gap: 8px; margin-bottom: 24px; font-size: 14px; }
        .breadcrumb a { color: var(--gray-500); text-decoration: none; }
        .breadcrumb a:hover { color: var(--primary); }
        .breadcrumb span { color: var(--gray-500); }
        .product-detail { display: grid; grid-template-columns: 1fr 1fr; gap: 48px; background: #fff; padding: 40px; border-radius: 16px; margin-bottom: 40px; }
        .product-gallery { position: relative; }
        .product-main-image { width: 100%; aspect-ratio: 1; object-fit: cover; border-radius: 12px; background: var(--gray-100); }
        .sale-badge { position: absolute; top: 16px; left: 16px; background: var(--danger); color: #fff; padding: 8px 16px; border-radius: 24px; font-weight: 600; }
        .product-info h1 { font-size: 32px; color: var(--gray-900); margin-bottom: 8px; }
        .product-category { color: var(--gray-500); margin-bottom: 20px; }
        .product-price { display: flex; align-items: center; gap: 16px; margin-bottom: 20px; }
        .price-current { font-size: 32px; font-weight: 700; color: var(--primary); }
        .price-original { font-size: 20px; color: var(--gray-500); text-decoration: line-through; }
        .discount-percent { background: #fee2e2; color: var(--danger); padding: 6px 12px; border-radius: 20px; font-weight: 600; font-size: 14px; }
        .product-description { line-height: 1.7; margin-bottom: 24px; color: var(--gray-700); }
        .product-stock { margin-bottom: 24px; }
        .stock-in { color: var(--success); font-weight: 500; }
        .stock-low { color: var(--danger); font-weight: 500; }
        .quantity-selector { display: flex; align-items: center; gap: 16px; margin-bottom: 24px; }
        .quantity-selector label { font-weight: 500; }
        .quantity-input { display: flex; align-items: center; border: 1px solid var(--gray-200); border-radius: 8px; overflow: hidden; }
        .quantity-input button { width: 44px; height: 44px; border: none; background: var(--gray-100); cursor: pointer; font-size: 18px; }
        .quantity-input button:hover { background: var(--gray-200); }
        .quantity-input input { width: 60px; height: 44px; border: none; text-align: center; font-size: 16px; font-weight: 500; }
        .btn { padding: 14px 28px; border-radius: 10px; text-decoration: none; font-weight: 600; cursor: pointer; border: none; transition: all 0.3s; display: inline-flex; align-items: center; gap: 10px; font-size: 16px; }
        .btn-primary { background: var(--primary); color: #fff; }
        .btn-primary:hover { background: var(--primary-dark); transform: translateY(-2px); }
        .btn-secondary { background: #fff; color: var(--primary); border: 2px solid var(--primary); }
        .action-buttons { display: flex; gap: 16px; }
        .product-meta { margin-top: 24px; padding-top: 24px; border-top: 1px solid var(--gray-200); }
        .meta-item { display: flex; gap: 8px; margin-bottom: 8px; font-size: 14px; }
        .meta-item span:first-child { color: var(--gray-500); min-width: 80px; }
        .section-title { font-size: 24px; color: var(--gray-900); margin-bottom: 24px; }
        .related-products { display: grid; grid-template-columns: repeat(4, 1fr); gap: 24px; }
        .product-card { background: #fff; border-radius: 12px; overflow: hidden; box-shadow: 0 1px 3px rgba(0,0,0,0.05); transition: transform 0.3s; }
        .product-card:hover { transform: translateY(-4px); }
        .product-card-image { position: relative; padding-top: 100%; background: var(--gray-100); }
        .product-card-image img { position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover; }
        .product-card-info { padding: 16px; }
        .product-card-name { font-weight: 600; margin-bottom: 8px; color: var(--gray-900); }
        .product-card-name a { color: inherit; text-decoration: none; }
        .product-card-price { color: var(--primary); font-weight: 700; }
        @media (max-width: 768px) {
            .product-detail { grid-template-columns: 1fr; padding: 20px; }
            .related-products { grid-template-columns: repeat(2, 1fr); }
            .action-buttons { flex-direction: column; }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <a href="../index.php" class="logo"><?php echo htmlspecialchars($settings['site_name']); ?></a>
            <nav class="nav">
                <a href="products.php">Shop</a>
                <a href="cart.php" class="cart-link">
                    <i class="fas fa-shopping-cart"></i>
                    <?php if ($cartCount > 0): ?>
                    <span class="cart-badge"><?php echo $cartCount; ?></span>
                    <?php endif; ?>
                </a>
                <?php if ($user): ?>
                <a href="dashboard.php">My Account</a>
                <a href="logout.php">Logout</a>
                <?php else: ?>
                <a href="login.php">Login</a>
                <?php endif; ?>
            </nav>
        </div>
    </header>

    <div class="container">
        <div class="breadcrumb">
            <a href="../index.php">Home</a>
            <span>/</span>
            <a href="products.php">Products</a>
            <?php if ($product['category_name']): ?>
            <span>/</span>
            <a href="products.php?category=<?php echo $product['category_id']; ?>"><?php echo htmlspecialchars($product['category_name']); ?></a>
            <?php endif; ?>
            <span>/</span>
            <span><?php echo htmlspecialchars($product['name']); ?></span>
        </div>

        <div class="product-detail">
            <div class="product-gallery">
                <?php if ($product['image']): ?>
                <img src="../uploads/products/<?php echo htmlspecialchars($product['image']); ?>" 
                     alt="<?php echo htmlspecialchars($product['name']); ?>" 
                     class="product-main-image">
                <?php else: ?>
                <div class="product-main-image" style="display: flex; align-items: center; justify-content: center;">
                    <i class="fas fa-image" style="font-size: 80px; color: var(--gray-200);"></i>
                </div>
                <?php endif; ?>
                <?php if ($product['sale_price']): ?>
                <span class="sale-badge">
                    <?php echo round((($product['price'] - $product['sale_price']) / $product['price']) * 100); ?>% OFF
                </span>
                <?php endif; ?>
            </div>

            <div class="product-info">
                <h1><?php echo htmlspecialchars($product['name']); ?></h1>
                <div class="product-category"><?php echo htmlspecialchars($product['category_name'] ?? 'General'); ?></div>
                
                <div class="product-price">
                    <span class="price-current"><?php echo formatPrice($product['sale_price'] ?? $product['price']); ?></span>
                    <?php if ($product['sale_price']): ?>
                    <span class="price-original"><?php echo formatPrice($product['price']); ?></span>
                    <span class="discount-percent">Save <?php echo round((($product['price'] - $product['sale_price']) / $product['price']) * 100); ?>%</span>
                    <?php endif; ?>
                </div>

                <?php if ($product['short_description']): ?>
                <p class="product-description"><?php echo htmlspecialchars($product['short_description']); ?></p>
                <?php endif; ?>

                <div class="product-stock">
                    <?php if ($product['stock'] > 0): ?>
                    <span class="<?php echo $product['stock'] <= 5 ? 'stock-low' : 'stock-in'; ?>">
                        <i class="fas fa-check-circle"></i>
                        <?php echo $product['stock'] <= 5 ? 'Only ' . $product['stock'] . ' left in stock!' : 'In Stock (' . $product['stock'] . ' available)'; ?>
                    </span>
                    <?php else: ?>
                    <span class="stock-low"><i class="fas fa-times-circle"></i> Out of Stock</span>
                    <?php endif; ?>
                </div>

                <?php if ($product['stock'] > 0): ?>
                <form action="cart.php" method="GET">
                    <input type="hidden" name="add" value="<?php echo $product['id']; ?>">
                    <div class="quantity-selector">
                        <label>Quantity:</label>
                        <div class="quantity-input">
                            <button type="button" onclick="changeQty(-1)">-</button>
                            <input type="number" name="qty" id="quantity" value="1" min="1" max="<?php echo $product['stock']; ?>">
                            <button type="button" onclick="changeQty(1)">+</button>
                        </div>
                    </div>
                    <div class="action-buttons">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-cart-plus"></i> Add to Cart
                        </button>
                        <a href="checkout.php?buy=<?php echo $product['id']; ?>" class="btn btn-secondary">
                            <i class="fas fa-bolt"></i> Buy Now
                        </a>
                    </div>
                </form>
                <?php else: ?>
                <div class="action-buttons">
                    <button class="btn btn-primary" disabled style="background: var(--gray-200); color: var(--gray-500);">Out of Stock</button>
                </div>
                <?php endif; ?>

                <div class="product-meta">
                    <?php if ($product['sku']): ?>
                    <div class="meta-item">
                        <span>SKU:</span>
                        <span><?php echo htmlspecialchars($product['sku']); ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="meta-item">
                        <span>Category:</span>
                        <span><?php echo htmlspecialchars($product['category_name'] ?? 'Uncategorized'); ?></span>
                    </div>
                </div>
            </div>
        </div>

        <?php if ($product['description']): ?>
        <div style="background: #fff; padding: 30px; border-radius: 12px; margin-bottom: 40px;">
            <h2 class="section-title">Description</h2>
            <div style="line-height: 1.8; color: var(--gray-700);">
                <?php echo nl2br(htmlspecialchars($product['description'])); ?>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($relatedProducts)): ?>
        <h2 class="section-title">Related Products</h2>
        <div class="related-products">
            <?php foreach ($relatedProducts as $related): ?>
            <div class="product-card">
                <div class="product-card-image">
                    <?php if ($related['image']): ?>
                    <a href="product-view.php?slug=<?php echo htmlspecialchars($related['slug']); ?>">
                        <img src="../uploads/products/<?php echo htmlspecialchars($related['image']); ?>" 
                             alt="<?php echo htmlspecialchars($related['name']); ?>">
                    </a>
                    <?php endif; ?>
                </div>
                <div class="product-card-info">
                    <h3 class="product-card-name">
                        <a href="product-view.php?slug=<?php echo htmlspecialchars($related['slug']); ?>">
                            <?php echo htmlspecialchars($related['name']); ?>
                        </a>
                    </h3>
                    <div class="product-card-price"><?php echo formatPrice($related['sale_price'] ?? $related['price']); ?></div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>

    <script>
        function changeQty(delta) {
            const input = document.getElementById('quantity');
            const newVal = parseInt(input.value) + delta;
            if (newVal >= 1 && newVal <= parseInt(input.max)) {
                input.value = newVal;
            }
        }
    </script>
</body>
</html>
