<?php
/**
 * User Products Listing
 */
define('ECOMMERCE_ACCESS', true);
require_once '../config/db.php';
require_once '../config/auth.php';

$settings = getSettings();
$user = getCurrentUser();

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 12;
$offset = ($page - 1) * $limit;

// Filters
$search = sanitize($_GET['search'] ?? '');
$category = intval($_GET['category'] ?? 0);
$sort = sanitize($_GET['sort'] ?? 'newest');

$where = "p.status = 'active'";
$params = [];

if ($search) {
    $where .= " AND (p.name LIKE ? OR p.description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}
if ($category) {
    $where .= " AND p.category_id = ?";
    $params[] = $category;
}

// Sorting
$orderBy = "p.created_at DESC";
switch ($sort) {
    case 'price_low': $orderBy = "COALESCE(p.sale_price, p.price) ASC"; break;
    case 'price_high': $orderBy = "COALESCE(p.sale_price, p.price) DESC"; break;
    case 'name': $orderBy = "p.name ASC"; break;
    case 'popular': $orderBy = "p.views DESC"; break;
}

// Get total count
$totalCount = fetchOne("SELECT COUNT(*) as count FROM products p WHERE $where", $params)['count'];
$totalPages = ceil($totalCount / $limit);

// Get products
$products = fetchAll("
    SELECT p.*, c.name as category_name 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    WHERE $where 
    ORDER BY $orderBy 
    LIMIT $limit OFFSET $offset
", $params);

// Get categories
$categories = fetchAll("SELECT * FROM categories WHERE status = 1 ORDER BY name");

// Cart count
$cartCount = 0;
if ($user) {
    $cartCount = fetchOne("SELECT COUNT(*) as count FROM cart WHERE user_id = ?", [$user['id']])['count'];
} else {
    $cartCount = fetchOne("SELECT COUNT(*) as count FROM cart WHERE session_id = ?", [session_id()])['count'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products - <?php echo htmlspecialchars($settings['site_name']); ?></title>
    <link rel="stylesheet" href="../assets/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root { --primary: #0ea5e9; --primary-dark: #0284c7; --success: #10b981; --danger: #ef4444; --gray-100: #f1f5f9; --gray-200: #e2e8f0; --gray-500: #64748b; --gray-700: #334155; --gray-900: #0f172a; }
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', system-ui, sans-serif; background: var(--gray-100); color: var(--gray-700); }
        .header { background: #fff; padding: 16px 0; box-shadow: 0 1px 3px rgba(0,0,0,0.1); position: sticky; top: 0; z-index: 100; }
        .header-content { max-width: 1200px; margin: 0 auto; padding: 0 20px; display: flex; justify-content: space-between; align-items: center; }
        .logo { font-size: 24px; font-weight: 700; color: var(--primary); text-decoration: none; }
        .nav { display: flex; gap: 24px; align-items: center; }
        .nav a { color: var(--gray-700); text-decoration: none; font-weight: 500; transition: color 0.3s; }
        .nav a:hover { color: var(--primary); }
        .nav .cart-link { position: relative; }
        .cart-badge { position: absolute; top: -8px; right: -12px; background: var(--primary); color: #fff; font-size: 11px; padding: 2px 6px; border-radius: 10px; }
        .container { max-width: 1200px; margin: 0 auto; padding: 30px 20px; }
        .page-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px; flex-wrap: wrap; gap: 16px; }
        .page-header h1 { font-size: 28px; color: var(--gray-900); }
        .filters { display: flex; gap: 16px; flex-wrap: wrap; margin-bottom: 24px; background: #fff; padding: 20px; border-radius: 12px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); }
        .filters input, .filters select { padding: 10px 14px; border: 1px solid var(--gray-200); border-radius: 8px; font-size: 14px; }
        .filters input { flex: 1; min-width: 200px; }
        .btn { padding: 10px 20px; border-radius: 8px; text-decoration: none; font-weight: 500; cursor: pointer; border: none; transition: all 0.3s; display: inline-flex; align-items: center; gap: 8px; }
        .btn-primary { background: var(--primary); color: #fff; }
        .btn-primary:hover { background: var(--primary-dark); }
        .products-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(260px, 1fr)); gap: 24px; }
        .product-card { background: #fff; border-radius: 12px; overflow: hidden; box-shadow: 0 1px 3px rgba(0,0,0,0.05); transition: transform 0.3s, box-shadow 0.3s; }
        .product-card:hover { transform: translateY(-4px); box-shadow: 0 10px 30px rgba(0,0,0,0.1); }
        .product-image { position: relative; padding-top: 100%; background: var(--gray-100); }
        .product-image img { position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover; }
        .product-image .no-image { position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); color: var(--gray-500); font-size: 48px; }
        .sale-badge { position: absolute; top: 12px; left: 12px; background: var(--danger); color: #fff; padding: 4px 10px; border-radius: 20px; font-size: 12px; font-weight: 600; }
        .product-info { padding: 20px; }
        .product-category { font-size: 12px; color: var(--gray-500); text-transform: uppercase; margin-bottom: 8px; }
        .product-name { font-size: 16px; font-weight: 600; color: var(--gray-900); margin-bottom: 8px; display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow: hidden; }
        .product-name a { color: inherit; text-decoration: none; }
        .product-price { display: flex; align-items: center; gap: 10px; margin-bottom: 12px; }
        .price-current { font-size: 18px; font-weight: 700; color: var(--primary); }
        .price-original { font-size: 14px; color: var(--gray-500); text-decoration: line-through; }
        .product-stock { font-size: 13px; margin-bottom: 12px; }
        .stock-in { color: var(--success); }
        .stock-low { color: var(--danger); }
        .btn-cart { width: 100%; justify-content: center; }
        .pagination { display: flex; justify-content: center; gap: 8px; margin-top: 40px; }
        .pagination a, .pagination span { padding: 10px 16px; border-radius: 8px; text-decoration: none; color: var(--gray-700); background: #fff; border: 1px solid var(--gray-200); }
        .pagination a:hover { background: var(--primary); color: #fff; border-color: var(--primary); }
        .pagination .active { background: var(--primary); color: #fff; border-color: var(--primary); }
        .empty-state { text-align: center; padding: 60px 20px; background: #fff; border-radius: 12px; }
        .empty-state i { font-size: 64px; color: var(--gray-200); margin-bottom: 20px; }
        .empty-state p { color: var(--gray-500); font-size: 18px; }
        @media (max-width: 768px) {
            .nav { gap: 16px; }
            .filters { flex-direction: column; }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <a href="../index.php" class="logo"><?php echo htmlspecialchars($settings['site_name']); ?></a>
            <nav class="nav">
                <a href="products.php">Shop</a>
                <a href="cart.php" class="cart-link">
                    <i class="fas fa-shopping-cart"></i>
                    <?php if ($cartCount > 0): ?>
                    <span class="cart-badge"><?php echo $cartCount; ?></span>
                    <?php endif; ?>
                </a>
                <?php if ($user): ?>
                <a href="dashboard.php">My Account</a>
                <a href="logout.php">Logout</a>
                <?php else: ?>
                <a href="login.php">Login</a>
                <?php endif; ?>
            </nav>
        </div>
    </header>

    <div class="container">
        <div class="page-header">
            <h1>Products <?php if ($search): ?> - Search: "<?php echo htmlspecialchars($search); ?>"<?php endif; ?></h1>
            <span><?php echo number_format($totalCount); ?> products found</span>
        </div>

        <form method="GET" class="filters">
            <input type="text" name="search" placeholder="Search products..." value="<?php echo htmlspecialchars($search); ?>">
            <select name="category">
                <option value="">All Categories</option>
                <?php foreach ($categories as $cat): ?>
                <option value="<?php echo $cat['id']; ?>" <?php echo $category == $cat['id'] ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($cat['name']); ?>
                </option>
                <?php endforeach; ?>
            </select>
            <select name="sort">
                <option value="newest" <?php echo $sort === 'newest' ? 'selected' : ''; ?>>Newest</option>
                <option value="price_low" <?php echo $sort === 'price_low' ? 'selected' : ''; ?>>Price: Low to High</option>
                <option value="price_high" <?php echo $sort === 'price_high' ? 'selected' : ''; ?>>Price: High to Low</option>
                <option value="name" <?php echo $sort === 'name' ? 'selected' : ''; ?>>Name</option>
                <option value="popular" <?php echo $sort === 'popular' ? 'selected' : ''; ?>>Popular</option>
            </select>
            <button type="submit" class="btn btn-primary">Filter</button>
        </form>

        <?php if (empty($products)): ?>
        <div class="empty-state">
            <i class="fas fa-search"></i>
            <p>No products found matching your criteria.</p>
        </div>
        <?php else: ?>
        <div class="products-grid">
            <?php foreach ($products as $product): ?>
            <div class="product-card">
                <div class="product-image">
                    <?php if ($product['image']): ?>
                    <a href="product-view.php?slug=<?php echo htmlspecialchars($product['slug']); ?>">
                        <img src="../uploads/products/<?php echo htmlspecialchars($product['image']); ?>" alt="<?php echo htmlspecialchars($product['name']); ?>">
                    </a>
                    <?php else: ?>
                    <div class="no-image"><i class="fas fa-image"></i></div>
                    <?php endif; ?>
                    <?php if ($product['sale_price']): ?>
                    <span class="sale-badge">Sale</span>
                    <?php endif; ?>
                </div>
                <div class="product-info">
                    <div class="product-category"><?php echo htmlspecialchars($product['category_name'] ?? 'General'); ?></div>
                    <h3 class="product-name">
                        <a href="product-view.php?slug=<?php echo htmlspecialchars($product['slug']); ?>">
                            <?php echo htmlspecialchars($product['name']); ?>
                        </a>
                    </h3>
                    <div class="product-price">
                        <span class="price-current"><?php echo formatPrice($product['sale_price'] ?? $product['price']); ?></span>
                        <?php if ($product['sale_price']): ?>
                        <span class="price-original"><?php echo formatPrice($product['price']); ?></span>
                        <?php endif; ?>
                    </div>
                    <div class="product-stock">
                        <?php if ($product['stock'] > 0): ?>
                        <span class="<?php echo $product['stock'] <= 5 ? 'stock-low' : 'stock-in'; ?>">
                            <?php echo $product['stock'] <= 5 ? 'Only ' . $product['stock'] . ' left!' : 'In Stock'; ?>
                        </span>
                        <?php else: ?>
                        <span class="stock-low">Out of Stock</span>
                        <?php endif; ?>
                    </div>
                    <?php if ($product['stock'] > 0): ?>
                    <a href="cart.php?add=<?php echo $product['id']; ?>" class="btn btn-primary btn-cart">
                        <i class="fas fa-cart-plus"></i> Add to Cart
                    </a>
                    <?php else: ?>
                    <button class="btn btn-cart" disabled style="background: var(--gray-200); color: var(--gray-500);">Out of Stock</button>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>

        <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
            <a href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&category=<?php echo $category; ?>&sort=<?php echo urlencode($sort); ?>">&laquo; Prev</a>
            <?php endif; ?>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
            <?php if ($i == $page): ?>
            <span class="active"><?php echo $i; ?></span>
            <?php else: ?>
            <a href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&category=<?php echo $category; ?>&sort=<?php echo urlencode($sort); ?>"><?php echo $i; ?></a>
            <?php endif; ?>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
            <a href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&category=<?php echo $category; ?>&sort=<?php echo urlencode($sort); ?>">Next &raquo;</a>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        <?php endif; ?>
    </div>
</body>
</html>
